"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = __importDefault(require("express"));
const cors_1 = __importDefault(require("cors"));
const express_session_1 = __importDefault(require("express-session"));
const api_1 = require("./db/api"); // Veritabanı API'mizi import ediyoruz
const app = (0, express_1.default)();
const port = process.env.PORT || 3001; // cPanel portunu kullan
app.use((0, cors_1.default)({
    origin: ['http://localhost:5173', 'https://alanadiniz.com'], // Alan adınızı ekleyin
    credentials: true
}));
app.use(express_1.default.json());
// Session middleware 
app.use((0, express_session_1.default)({
    secret: process.env.SESSION_SECRET || 'your-session-secret-change-in-production',
    resave: false,
    saveUninitialized: false,
    cookie: {
        secure: false, // Set to true in production with HTTPS
        maxAge: 24 * 60 * 60 * 1000 // 24 hours
    }
}));
// Authentication middleware
const authenticateToken = async (req, res, next) => {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];
    if (!token) {
        return res.status(401).json({ message: 'Yetkilendirme token\'ı gerekli' });
    }
    try {
        const user = await api_1.db.verifyToken(token);
        if (!user) {
            return res.status(403).json({ message: 'Geçersiz veya süresi dolmuş token' });
        }
        req.user = user;
        next();
    }
    catch (error) {
        return res.status(403).json({ message: 'Geçersiz token' });
    }
};
// --- Authentication Endpoints ---
// User login
app.post('/api/auth/login', async (req, res) => {
    try {
        const loginData = req.body;
        if (!loginData.username || !loginData.password) {
            return res.status(400).json({
                success: false,
                message: 'Kullanıcı adı ve şifre zorunludur'
            });
        }
        const result = await api_1.db.authenticateUser(loginData);
        if (result.user && result.token) {
            // Set session
            req.session.userId = result.user.id;
            req.session.user = result.user;
            res.json({
                success: true,
                message: result.message,
                user: result.user,
                token: result.token
            });
        }
        else {
            res.status(401).json({
                success: false,
                message: result.message
            });
        }
    }
    catch (error) {
        console.error('Login Error:', error);
        res.status(500).json({
            success: false,
            message: 'Giriş yapılırken bir hata oluştu'
        });
    }
});
// User logout
app.post('/api/auth/logout', authenticateToken, (req, res) => {
    req.session.destroy((err) => {
        if (err) {
            return res.status(500).json({
                success: false,
                message: 'Çıkış yapılırken bir hata oluştu'
            });
        }
        res.json({
            success: true,
            message: 'Başarıyla çıkış yapıldı'
        });
    });
});
// Get current user
app.get('/api/auth/me', authenticateToken, (req, res) => {
    res.json({
        success: true,
        user: req.user
    });
});
// --- API Endpoints ---
// Kategorileri getir
app.get('/api/categories', async (req, res) => {
    try {
        const categories = await api_1.db.getCategories();
        res.json(categories);
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Kategoriler alınırken bir hata oluştu.' });
    }
});
// Yeni kategori oluştur
app.post('/api/categories', async (req, res) => {
    try {
        const categoryData = req.body;
        const newCategory = await api_1.db.createCategory(categoryData);
        res.json({ message: 'Kategori başarıyla oluşturuldu', category: newCategory });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Kategori oluşturulurken bir hata oluştu.' });
    }
});
// Kategori güncelle
app.put('/api/categories/:id', async (req, res) => {
    try {
        const { id } = req.params;
        const updates = req.body;
        await api_1.db.updateCategory(id, updates);
        const categories = await api_1.db.getCategories();
        const category = categories.find(c => c.id === id);
        if (category) {
            res.json({ message: 'Kategori başarıyla güncellendi', category });
        }
        else {
            res.status(404).json({ message: 'Kategori bulunamadı' });
        }
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Kategori güncellenirken bir hata oluştu.' });
    }
});
// Kategori sil
app.delete('/api/categories/:id', async (req, res) => {
    try {
        const { id } = req.params;
        await api_1.db.deleteCategory(id);
        res.json({ message: 'Kategori başarıyla silindi' });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Kategori silinirken bir hata oluştu.' });
    }
});
// Adayları getir (filtreleme ile)
app.get('/api/candidates', async (req, res) => {
    try {
        // Filtre parametrelerini req.query'den al
        const filters = {
            status: req.query.status,
            category: req.query.category,
            city: req.query.city,
            minHeight: req.query.minHeight ? parseInt(req.query.minHeight) : undefined,
            maxHeight: req.query.maxHeight ? parseInt(req.query.maxHeight) : undefined,
            minWeight: req.query.minWeight ? parseInt(req.query.minWeight) : undefined,
            maxWeight: req.query.maxWeight ? parseInt(req.query.maxWeight) : undefined,
            minAge: req.query.minAge ? parseInt(req.query.minAge) : undefined,
            maxAge: req.query.maxAge ? parseInt(req.query.maxAge) : undefined,
            languages: req.query.languages ? req.query.languages.split(',') : undefined,
            eyeColor: req.query.eyeColor,
            hairColor: req.query.hairColor
        };
        // undefined değerleri filtreleyelim
        Object.keys(filters).forEach(key => {
            if (filters[key] === undefined) {
                delete filters[key];
            }
        });
        console.log('API Filters:', filters);
        const candidates = await api_1.db.getCandidates(filters);
        res.json(candidates);
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Adaylar alınırken bir hata oluştu.' });
    }
});
// Yeni aday oluştur
app.post('/api/candidates', async (req, res) => {
    try {
        const candidateData = req.body;
        const newCandidate = await api_1.db.createCandidate(candidateData);
        res.json({ message: 'Aday başarıyla oluşturuldu', candidate: newCandidate });
    }
    catch (error) {
        console.error('API Error:', error);
        // Email duplicate hatası özel handling
        if (error.code === 'ER_DUP_ENTRY' && error.sqlMessage.includes('email')) {
            return res.status(400).json({
                message: 'Bu email adresiyle zaten bir kayıt bulunuyor. Lütfen farklı bir email adresi kullanın.',
                field: 'email',
                duplicate: true
            });
        }
        // Diğer hatalar
        res.status(500).json({ message: 'Aday oluşturulurken bir hata oluştu.' });
    }
});
// Müşterileri getir
app.get('/api/customers', async (req, res) => {
    try {
        const customers = await api_1.db.getCustomers();
        res.json(customers);
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Müşteriler alınırken bir hata oluştu.' });
    }
});
// Seçimleri getir
app.get('/api/selections', async (req, res) => {
    try {
        const selections = await api_1.db.getSelections();
        res.json(selections);
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Seçimler alınırken bir hata oluştu.' });
    }
});
// Yeni seçim oluştur
app.post('/api/selections', async (req, res) => {
    try {
        const selectionData = req.body;
        const newSelection = await api_1.db.createSelection(selectionData);
        res.json({ message: 'Seçim başarıyla oluşturuldu', selection: newSelection });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Seçim oluşturulurken bir hata oluştu.' });
    }
});
// Bir seçime ait adayları getir
app.get('/api/selections/:id/candidates', async (req, res) => {
    try {
        const { id } = req.params;
        const selectionCandidates = await api_1.db.getSelectionCandidatesBySelectionId(id);
        res.json(selectionCandidates);
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Seçim adayları alınırken bir hata oluştu.' });
    }
});
// Bir seçime aday ekle
app.post('/api/selections/:id/candidates', async (req, res) => {
    try {
        const { id } = req.params;
        const candidateData = req.body;
        // SelectionCandidate oluştur
        const newSelectionCandidate = await api_1.db.createSelectionCandidate(id, candidateData);
        res.json({ message: 'Aday seçime başarıyla eklendi', selectionCandidate: newSelectionCandidate });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Aday seçime eklenirken bir hata oluştu.' });
    }
});
// Adayı güncelle
app.put('/api/candidates/:id', async (req, res) => {
    try {
        const { id } = req.params;
        const updates = req.body;
        await api_1.db.updateCandidate(id, updates);
        const candidate = await api_1.db.getCandidateById(id);
        if (candidate) {
            res.json({ message: 'Aday başarıyla güncellendi', candidate });
        }
        else {
            res.status(404).json({ message: 'Aday bulunamadı' });
        }
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Aday güncellenirken bir hata oluştu.' });
    }
});
// Bir seçimden aday çıkar
app.delete('/api/selections/:id/candidates/:candidateId', async (req, res) => {
    try {
        const { id, candidateId } = req.params;
        await api_1.db.removeSelectionCandidate(id, candidateId);
        res.json({ message: 'Aday seçimden başarıyla çıkarıldı' });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Aday seçimden çıkarılırken bir hata oluştu.' });
    }
});
// Bir seçimi sil
app.delete('/api/selections/:id', async (req, res) => {
    try {
        const { id } = req.params;
        console.log('Deleting selection with id:', id);
        await api_1.db.deleteSelection(id);
        res.json({ message: 'Seçim başarıyla silindi' });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Seçim silinirken bir hata oluştu.' });
    }
});
// --- Şehirler API Endpoint'leri ---
// Tüm şehirleri getir
app.get('/api/cities', async (req, res) => {
    try {
        const cities = await api_1.db.getCities();
        res.json(cities);
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Şehirler alınırken bir hata oluştu.' });
    }
});
// Yeni şehir ekle
app.post('/api/cities', async (req, res) => {
    try {
        const cityData = req.body;
        const newCity = await api_1.db.addCity(cityData);
        res.json({ message: 'Şehir başarıyla eklendi', city: newCity });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Şehir eklenirken bir hata oluştu.' });
    }
});
// Şehir güncelle
app.put('/api/cities/:id', async (req, res) => {
    try {
        const { id } = req.params;
        const updates = req.body;
        await api_1.db.updateCity(id, updates);
        res.json({ message: 'Şehir başarıyla güncellendi' });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Şehir güncellenirken bir hata oluştu.' });
    }
});
// Şehir sil
app.delete('/api/cities/:id', async (req, res) => {
    try {
        const { id } = req.params;
        await api_1.db.deleteCity(id);
        res.json({ message: 'Şehir başarıyla silindi' });
    }
    catch (error) {
        console.error('API Error:', error);
        res.status(500).json({ message: 'Şehir silinirken bir hata oluştu.' });
    }
});
app.listen(port, () => {
    console.log(`Backend server http://localhost:${port} adresinde çalışıyor`);
});
